// MyNetworkTools Chrome Extension - Tool Implementations
const API_BASE = 'https://mynetworktools.com/api';

const Tools = {
  // ==================== PASSWORD GENERATOR ====================
  'password-gen': {
    title: 'Password Generator',
    render: () => `
      <div class="tool-section">
        <div class="form-group">
          <label>Length: <span id="lengthValue">16</span></label>
          <input type="range" id="passLength" min="8" max="64" value="16">
        </div>
        <div class="checkbox-group">
          <label><input type="checkbox" id="passUpper" checked> Uppercase (A-Z)</label>
          <label><input type="checkbox" id="passLower" checked> Lowercase (a-z)</label>
          <label><input type="checkbox" id="passNumbers" checked> Numbers (0-9)</label>
          <label><input type="checkbox" id="passSymbols" checked> Symbols (!@#$%)</label>
        </div>
        <button class="btn-primary" id="generatePass">Generate Password</button>
        <div class="result-box">
          <input type="text" id="passResult" readonly placeholder="Generated password will appear here">
          <button class="btn-copy" data-copy="passResult">Copy</button>
        </div>
        <div id="passStrength" class="strength-meter"></div>
      </div>
    `,
    init: () => {
      const lengthSlider = document.getElementById('passLength');
      const lengthValue = document.getElementById('lengthValue');
      lengthSlider.oninput = () => lengthValue.textContent = lengthSlider.value;

      document.getElementById('generatePass').onclick = () => {
        const len = parseInt(lengthSlider.value);
        let chars = '';
        if (document.getElementById('passUpper').checked) chars += 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
        if (document.getElementById('passLower').checked) chars += 'abcdefghijklmnopqrstuvwxyz';
        if (document.getElementById('passNumbers').checked) chars += '0123456789';
        if (document.getElementById('passSymbols').checked) chars += '!@#$%^&*()_+-=[]{}|;:,.<>?';

        if (!chars) { alert('Select at least one character type'); return; }

        const array = new Uint32Array(len);
        crypto.getRandomValues(array);
        const password = Array.from(array, x => chars[x % chars.length]).join('');
        document.getElementById('passResult').value = password;

        // Strength indicator
        const strength = len >= 20 && chars.length > 60 ? 'Strong' : len >= 12 ? 'Medium' : 'Weak';
        const color = strength === 'Strong' ? '#4ade80' : strength === 'Medium' ? '#fbbf24' : '#ef4444';
        document.getElementById('passStrength').innerHTML = `<span style="color:${color}">Strength: ${strength}</span>`;
      };
    }
  },

  // ==================== HASH GENERATOR ====================
  'hash-gen': {
    title: 'Hash Generator',
    render: () => `
      <div class="tool-section">
        <div class="form-group">
          <label>Text to Hash</label>
          <textarea id="hashInput" rows="3" placeholder="Enter text to hash..."></textarea>
        </div>
        <button class="btn-primary" id="generateHash">Generate Hashes</button>
        <div class="results-list" id="hashResults"></div>
      </div>
    `,
    init: () => {
      document.getElementById('generateHash').onclick = async () => {
        const text = document.getElementById('hashInput').value;
        if (!text) { alert('Enter text to hash'); return; }

        const encoder = new TextEncoder();
        const data = encoder.encode(text);

        const algorithms = ['SHA-1', 'SHA-256', 'SHA-384', 'SHA-512'];
        const results = document.getElementById('hashResults');
        results.innerHTML = '<div class="loading">Generating...</div>';

        let html = '';
        for (const algo of algorithms) {
          const hashBuffer = await crypto.subtle.digest(algo, data);
          const hashArray = Array.from(new Uint8Array(hashBuffer));
          const hashHex = hashArray.map(b => b.toString(16).padStart(2, '0')).join('');
          html += `
            <div class="result-item">
              <label>${algo}</label>
              <div class="result-box">
                <input type="text" value="${hashHex}" readonly class="hash-result">
                <button class="btn-copy" data-value="${hashHex}">Copy</button>
              </div>
            </div>
          `;
        }

        // MD5 (simple implementation for extension)
        const md5 = simpleMD5(text);
        html = `
          <div class="result-item">
            <label>MD5</label>
            <div class="result-box">
              <input type="text" value="${md5}" readonly class="hash-result">
              <button class="btn-copy" data-value="${md5}">Copy</button>
            </div>
          </div>
        ` + html;

        results.innerHTML = html;
        setupCopyButtons();
      };
    }
  },

  // ==================== BASE64 ====================
  'base64': {
    title: 'Base64 Encode/Decode',
    render: () => `
      <div class="tool-section">
        <div class="form-group">
          <label>Input</label>
          <textarea id="base64Input" rows="4" placeholder="Enter text..."></textarea>
        </div>
        <div class="btn-group">
          <button class="btn-primary" id="base64Encode">Encode</button>
          <button class="btn-secondary" id="base64Decode">Decode</button>
        </div>
        <div class="form-group">
          <label>Output</label>
          <textarea id="base64Output" rows="4" readonly></textarea>
        </div>
        <button class="btn-copy" data-copy="base64Output">Copy Result</button>
      </div>
    `,
    init: () => {
      document.getElementById('base64Encode').onclick = () => {
        const input = document.getElementById('base64Input').value;
        document.getElementById('base64Output').value = btoa(unescape(encodeURIComponent(input)));
      };
      document.getElementById('base64Decode').onclick = () => {
        try {
          const input = document.getElementById('base64Input').value;
          document.getElementById('base64Output').value = decodeURIComponent(escape(atob(input)));
        } catch (e) {
          document.getElementById('base64Output').value = 'Error: Invalid Base64 string';
        }
      };
    }
  },

  // ==================== TEXT ENCRYPTION ====================
  'encryption': {
    title: 'Text Encryption (AES)',
    render: () => `
      <div class="tool-section">
        <div class="form-group">
          <label>Text</label>
          <textarea id="encryptInput" rows="3" placeholder="Enter text..."></textarea>
        </div>
        <div class="form-group">
          <label>Password/Key</label>
          <input type="password" id="encryptKey" placeholder="Enter encryption key">
        </div>
        <div class="btn-group">
          <button class="btn-primary" id="encryptBtn">Encrypt</button>
          <button class="btn-secondary" id="decryptBtn">Decrypt</button>
        </div>
        <div class="form-group">
          <label>Result</label>
          <textarea id="encryptOutput" rows="3" readonly></textarea>
        </div>
        <button class="btn-copy" data-copy="encryptOutput">Copy Result</button>
      </div>
    `,
    init: () => {
      document.getElementById('encryptBtn').onclick = async () => {
        const text = document.getElementById('encryptInput').value;
        const key = document.getElementById('encryptKey').value;
        if (!text || !key) { alert('Enter text and key'); return; }

        const encrypted = await encryptAES(text, key);
        document.getElementById('encryptOutput').value = encrypted;
      };
      document.getElementById('decryptBtn').onclick = async () => {
        const text = document.getElementById('encryptInput').value;
        const key = document.getElementById('encryptKey').value;
        if (!text || !key) { alert('Enter text and key'); return; }

        try {
          const decrypted = await decryptAES(text, key);
          document.getElementById('encryptOutput').value = decrypted;
        } catch (e) {
          document.getElementById('encryptOutput').value = 'Error: Decryption failed. Check your key.';
        }
      };
    }
  },

  // ==================== JSON FORMATTER ====================
  'json-formatter': {
    title: 'JSON Formatter',
    render: () => `
      <div class="tool-section">
        <div class="form-group">
          <label>JSON Input</label>
          <textarea id="jsonInput" rows="6" placeholder='{"key": "value"}'></textarea>
        </div>
        <div class="btn-group">
          <button class="btn-primary" id="formatJson">Format</button>
          <button class="btn-secondary" id="minifyJson">Minify</button>
          <button class="btn-secondary" id="validateJson">Validate</button>
        </div>
        <div class="form-group">
          <label>Output</label>
          <textarea id="jsonOutput" rows="6" readonly></textarea>
        </div>
        <button class="btn-copy" data-copy="jsonOutput">Copy Result</button>
      </div>
    `,
    init: () => {
      document.getElementById('formatJson').onclick = () => {
        try {
          const json = JSON.parse(document.getElementById('jsonInput').value);
          document.getElementById('jsonOutput').value = JSON.stringify(json, null, 2);
        } catch (e) {
          document.getElementById('jsonOutput').value = 'Error: ' + e.message;
        }
      };
      document.getElementById('minifyJson').onclick = () => {
        try {
          const json = JSON.parse(document.getElementById('jsonInput').value);
          document.getElementById('jsonOutput').value = JSON.stringify(json);
        } catch (e) {
          document.getElementById('jsonOutput').value = 'Error: ' + e.message;
        }
      };
      document.getElementById('validateJson').onclick = () => {
        try {
          JSON.parse(document.getElementById('jsonInput').value);
          document.getElementById('jsonOutput').value = '✓ Valid JSON';
        } catch (e) {
          document.getElementById('jsonOutput').value = '✗ Invalid: ' + e.message;
        }
      };
    }
  },

  // ==================== UUID GENERATOR ====================
  'uuid-gen': {
    title: 'UUID Generator',
    render: () => `
      <div class="tool-section">
        <div class="form-group">
          <label>Number of UUIDs</label>
          <input type="number" id="uuidCount" value="5" min="1" max="100">
        </div>
        <button class="btn-primary" id="generateUuid">Generate UUIDs</button>
        <div class="form-group">
          <label>Generated UUIDs</label>
          <textarea id="uuidOutput" rows="8" readonly></textarea>
        </div>
        <button class="btn-copy" data-copy="uuidOutput">Copy All</button>
      </div>
    `,
    init: () => {
      document.getElementById('generateUuid').onclick = () => {
        const count = Math.min(100, Math.max(1, parseInt(document.getElementById('uuidCount').value) || 1));
        const uuids = [];
        for (let i = 0; i < count; i++) {
          uuids.push(crypto.randomUUID());
        }
        document.getElementById('uuidOutput').value = uuids.join('\n');
      };
      document.getElementById('generateUuid').click();
    }
  },

  // ==================== TIMESTAMP CONVERTER ====================
  'timestamp': {
    title: 'Timestamp Converter',
    render: () => `
      <div class="tool-section">
        <div class="result-item">
          <label>Current Timestamp</label>
          <div class="result-box">
            <input type="text" id="currentTs" readonly>
            <button class="btn-copy" data-copy="currentTs">Copy</button>
          </div>
        </div>
        <hr>
        <div class="form-group">
          <label>Unix Timestamp (seconds)</label>
          <input type="number" id="tsInput" placeholder="e.g., 1700000000">
        </div>
        <button class="btn-primary" id="convertTs">Convert to Date</button>
        <div class="result-item">
          <label>Date/Time</label>
          <input type="text" id="tsOutput" readonly>
        </div>
        <hr>
        <div class="form-group">
          <label>Date Input</label>
          <input type="datetime-local" id="dateInput">
        </div>
        <button class="btn-secondary" id="convertDate">Convert to Timestamp</button>
        <div class="result-item">
          <label>Unix Timestamp</label>
          <div class="result-box">
            <input type="text" id="dateOutput" readonly>
            <button class="btn-copy" data-copy="dateOutput">Copy</button>
          </div>
        </div>
      </div>
    `,
    init: () => {
      const updateCurrent = () => {
        document.getElementById('currentTs').value = Math.floor(Date.now() / 1000);
      };
      updateCurrent();
      setInterval(updateCurrent, 1000);

      document.getElementById('convertTs').onclick = () => {
        const ts = parseInt(document.getElementById('tsInput').value);
        if (isNaN(ts)) { alert('Enter a valid timestamp'); return; }
        const date = new Date(ts * 1000);
        document.getElementById('tsOutput').value = date.toLocaleString() + ' (Local)';
      };

      document.getElementById('convertDate').onclick = () => {
        const dateStr = document.getElementById('dateInput').value;
        if (!dateStr) { alert('Select a date'); return; }
        const ts = Math.floor(new Date(dateStr).getTime() / 1000);
        document.getElementById('dateOutput').value = ts;
      };
    }
  },

  // ==================== COLOR PICKER ====================
  'color-picker': {
    title: 'Color Picker',
    render: () => `
      <div class="tool-section">
        <div class="color-preview" id="colorPreview" style="background:#4a9eff"></div>
        <input type="color" id="colorInput" value="#4a9eff">
        <div class="results-list">
          <div class="result-item">
            <label>HEX</label>
            <div class="result-box">
              <input type="text" id="colorHex" value="#4a9eff" readonly>
              <button class="btn-copy" data-copy="colorHex">Copy</button>
            </div>
          </div>
          <div class="result-item">
            <label>RGB</label>
            <div class="result-box">
              <input type="text" id="colorRgb" readonly>
              <button class="btn-copy" data-copy="colorRgb">Copy</button>
            </div>
          </div>
          <div class="result-item">
            <label>HSL</label>
            <div class="result-box">
              <input type="text" id="colorHsl" readonly>
              <button class="btn-copy" data-copy="colorHsl">Copy</button>
            </div>
          </div>
        </div>
      </div>
    `,
    init: () => {
      const colorInput = document.getElementById('colorInput');
      const updateColor = () => {
        const hex = colorInput.value;
        document.getElementById('colorPreview').style.background = hex;
        document.getElementById('colorHex').value = hex.toUpperCase();

        const r = parseInt(hex.slice(1,3), 16);
        const g = parseInt(hex.slice(3,5), 16);
        const b = parseInt(hex.slice(5,7), 16);
        document.getElementById('colorRgb').value = `rgb(${r}, ${g}, ${b})`;

        const max = Math.max(r, g, b) / 255;
        const min = Math.min(r, g, b) / 255;
        const l = (max + min) / 2;
        let h = 0, s = 0;
        if (max !== min) {
          const d = max - min;
          s = l > 0.5 ? d / (2 - max - min) : d / (max + min);
          switch (Math.max(r, g, b)) {
            case r: h = ((g - b) / 255 / d + (g < b ? 6 : 0)); break;
            case g: h = ((b - r) / 255 / d + 2); break;
            case b: h = ((r - g) / 255 / d + 4); break;
          }
          h /= 6;
        }
        document.getElementById('colorHsl').value = `hsl(${Math.round(h*360)}, ${Math.round(s*100)}%, ${Math.round(l*100)}%)`;
      };
      colorInput.oninput = updateColor;
      updateColor();
    }
  },

  // ==================== REGEX TESTER ====================
  'regex-tester': {
    title: 'Regex Tester',
    render: () => `
      <div class="tool-section">
        <div class="form-group">
          <label>Pattern</label>
          <input type="text" id="regexPattern" placeholder="e.g., \\d+">
        </div>
        <div class="form-group">
          <label>Flags</label>
          <div class="checkbox-group inline">
            <label><input type="checkbox" id="flagG" checked> g</label>
            <label><input type="checkbox" id="flagI"> i</label>
            <label><input type="checkbox" id="flagM"> m</label>
          </div>
        </div>
        <div class="form-group">
          <label>Test String</label>
          <textarea id="regexInput" rows="4" placeholder="Enter test text..."></textarea>
        </div>
        <button class="btn-primary" id="testRegex">Test Regex</button>
        <div class="form-group">
          <label>Matches</label>
          <div id="regexOutput" class="matches-list"></div>
        </div>
      </div>
    `,
    init: () => {
      document.getElementById('testRegex').onclick = () => {
        const pattern = document.getElementById('regexPattern').value;
        const text = document.getElementById('regexInput').value;
        if (!pattern) { alert('Enter a pattern'); return; }

        let flags = '';
        if (document.getElementById('flagG').checked) flags += 'g';
        if (document.getElementById('flagI').checked) flags += 'i';
        if (document.getElementById('flagM').checked) flags += 'm';

        try {
          const regex = new RegExp(pattern, flags);
          const matches = text.match(regex);
          const output = document.getElementById('regexOutput');

          if (matches && matches.length > 0) {
            output.innerHTML = matches.map((m, i) =>
              `<div class="match-item"><span class="match-index">${i+1}</span> ${escapeHtml(m)}</div>`
            ).join('');
          } else {
            output.innerHTML = '<div class="no-matches">No matches found</div>';
          }
        } catch (e) {
          document.getElementById('regexOutput').innerHTML = `<div class="error">Error: ${e.message}</div>`;
        }
      };
    }
  },

  // ==================== LOREM IPSUM ====================
  'lorem-ipsum': {
    title: 'Lorem Ipsum Generator',
    render: () => `
      <div class="tool-section">
        <div class="form-group">
          <label>Paragraphs</label>
          <input type="number" id="loremCount" value="3" min="1" max="20">
        </div>
        <button class="btn-primary" id="generateLorem">Generate</button>
        <div class="form-group">
          <label>Output</label>
          <textarea id="loremOutput" rows="8" readonly></textarea>
        </div>
        <button class="btn-copy" data-copy="loremOutput">Copy Text</button>
      </div>
    `,
    init: () => {
      const loremText = "Lorem ipsum dolor sit amet, consectetur adipiscing elit. Sed do eiusmod tempor incididunt ut labore et dolore magna aliqua. Ut enim ad minim veniam, quis nostrud exercitation ullamco laboris nisi ut aliquip ex ea commodo consequat. Duis aute irure dolor in reprehenderit in voluptate velit esse cillum dolore eu fugiat nulla pariatur. Excepteur sint occaecat cupidatat non proident, sunt in culpa qui officia deserunt mollit anim id est laborum.";
      const sentences = [
        "Curabitur pretium tincidunt lacus.",
        "Nulla gravida orci a odio.",
        "Nullam varius, turpis et commodo pharetra.",
        "Est eros bibendum elit, nec luctus magna felis sollicitudin mauris.",
        "Integer in mauris eu nibh euismod gravida.",
        "Duis ac tellus et risus vulputate vehicula.",
        "Donec lobortis risus a elit.",
        "Etiam sollicitudin, ipsum eu pulvinar rutrum.",
        "Phasellus stus nibh tellus molestie nunc.",
        "Fusce suscipit varius mi."
      ];

      document.getElementById('generateLorem').onclick = () => {
        const count = Math.min(20, Math.max(1, parseInt(document.getElementById('loremCount').value) || 1));
        const paragraphs = [];
        for (let i = 0; i < count; i++) {
          if (i === 0) {
            paragraphs.push(loremText);
          } else {
            const shuffled = [...sentences].sort(() => Math.random() - 0.5);
            paragraphs.push(shuffled.slice(0, 4 + Math.floor(Math.random() * 3)).join(' '));
          }
        }
        document.getElementById('loremOutput').value = paragraphs.join('\n\n');
      };
      document.getElementById('generateLorem').click();
    }
  },

  // ==================== URL ENCODE/DECODE ====================
  'url-encode': {
    title: 'URL Encode/Decode',
    render: () => `
      <div class="tool-section">
        <div class="form-group">
          <label>Input</label>
          <textarea id="urlInput" rows="3" placeholder="Enter text or URL..."></textarea>
        </div>
        <div class="btn-group">
          <button class="btn-primary" id="urlEncode">Encode</button>
          <button class="btn-secondary" id="urlDecode">Decode</button>
        </div>
        <div class="form-group">
          <label>Output</label>
          <textarea id="urlOutput" rows="3" readonly></textarea>
        </div>
        <button class="btn-copy" data-copy="urlOutput">Copy Result</button>
      </div>
    `,
    init: () => {
      document.getElementById('urlEncode').onclick = () => {
        document.getElementById('urlOutput').value = encodeURIComponent(document.getElementById('urlInput').value);
      };
      document.getElementById('urlDecode').onclick = () => {
        try {
          document.getElementById('urlOutput').value = decodeURIComponent(document.getElementById('urlInput').value);
        } catch (e) {
          document.getElementById('urlOutput').value = 'Error: Invalid encoded string';
        }
      };
    }
  },

  // ==================== HTML ENCODE/DECODE ====================
  'html-encode': {
    title: 'HTML Encode/Decode',
    render: () => `
      <div class="tool-section">
        <div class="form-group">
          <label>Input</label>
          <textarea id="htmlInput" rows="3" placeholder="<div>Hello</div>"></textarea>
        </div>
        <div class="btn-group">
          <button class="btn-primary" id="htmlEncode">Encode</button>
          <button class="btn-secondary" id="htmlDecode">Decode</button>
        </div>
        <div class="form-group">
          <label>Output</label>
          <textarea id="htmlOutput" rows="3" readonly></textarea>
        </div>
        <button class="btn-copy" data-copy="htmlOutput">Copy Result</button>
      </div>
    `,
    init: () => {
      document.getElementById('htmlEncode').onclick = () => {
        const text = document.getElementById('htmlInput').value;
        document.getElementById('htmlOutput').value = text
          .replace(/&/g, '&amp;')
          .replace(/</g, '&lt;')
          .replace(/>/g, '&gt;')
          .replace(/"/g, '&quot;')
          .replace(/'/g, '&#039;');
      };
      document.getElementById('htmlDecode').onclick = () => {
        const text = document.getElementById('htmlInput').value;
        const doc = new DOMParser().parseFromString(text, 'text/html');
        document.getElementById('htmlOutput').value = doc.documentElement.textContent;
      };
    }
  },

  // ==================== WORD COUNTER ====================
  'word-counter': {
    title: 'Word Counter',
    render: () => `
      <div class="tool-section">
        <div class="form-group">
          <label>Text</label>
          <textarea id="wordInput" rows="6" placeholder="Enter or paste your text here..."></textarea>
        </div>
        <div class="stats-grid" id="wordStats">
          <div class="stat-item"><span class="stat-value" id="statChars">0</span><span class="stat-label">Characters</span></div>
          <div class="stat-item"><span class="stat-value" id="statWords">0</span><span class="stat-label">Words</span></div>
          <div class="stat-item"><span class="stat-value" id="statLines">0</span><span class="stat-label">Lines</span></div>
          <div class="stat-item"><span class="stat-value" id="statSentences">0</span><span class="stat-label">Sentences</span></div>
        </div>
      </div>
    `,
    init: () => {
      const input = document.getElementById('wordInput');
      const updateStats = () => {
        const text = input.value;
        document.getElementById('statChars').textContent = text.length;
        document.getElementById('statWords').textContent = text.trim() ? text.trim().split(/\s+/).length : 0;
        document.getElementById('statLines').textContent = text ? text.split('\n').length : 0;
        document.getElementById('statSentences').textContent = text.trim() ? (text.match(/[.!?]+/g) || []).length : 0;
      };
      input.oninput = updateStats;
    }
  },

  // ==================== CASE CONVERTER ====================
  'case-converter': {
    title: 'Case Converter',
    render: () => `
      <div class="tool-section">
        <div class="form-group">
          <label>Input Text</label>
          <textarea id="caseInput" rows="3" placeholder="Enter text to convert..."></textarea>
        </div>
        <div class="btn-group wrap">
          <button class="btn-small" data-case="upper">UPPERCASE</button>
          <button class="btn-small" data-case="lower">lowercase</button>
          <button class="btn-small" data-case="title">Title Case</button>
          <button class="btn-small" data-case="sentence">Sentence case</button>
          <button class="btn-small" data-case="camel">camelCase</button>
          <button class="btn-small" data-case="snake">snake_case</button>
        </div>
        <div class="form-group">
          <label>Output</label>
          <textarea id="caseOutput" rows="3" readonly></textarea>
        </div>
        <button class="btn-copy" data-copy="caseOutput">Copy Result</button>
      </div>
    `,
    init: () => {
      document.querySelectorAll('[data-case]').forEach(btn => {
        btn.onclick = () => {
          const text = document.getElementById('caseInput').value;
          let result = '';
          switch (btn.dataset.case) {
            case 'upper': result = text.toUpperCase(); break;
            case 'lower': result = text.toLowerCase(); break;
            case 'title': result = text.toLowerCase().replace(/\b\w/g, c => c.toUpperCase()); break;
            case 'sentence': result = text.toLowerCase().replace(/(^\w|[.!?]\s+\w)/g, c => c.toUpperCase()); break;
            case 'camel': result = text.toLowerCase().replace(/[^a-zA-Z0-9]+(.)/g, (m, c) => c.toUpperCase()); break;
            case 'snake': result = text.toLowerCase().replace(/\s+/g, '_').replace(/[^a-z0-9_]/g, ''); break;
          }
          document.getElementById('caseOutput').value = result;
        };
      });
    }
  },

  // ==================== DNS LOOKUP (API) ====================
  'dns-lookup': {
    title: 'DNS Lookup',
    render: () => `
      <div class="tool-section">
        <div class="form-group">
          <label>Domain Name</label>
          <input type="text" id="dnsInput" placeholder="example.com">
        </div>
        <button class="btn-primary" id="dnsLookup">Lookup DNS</button>
        <div id="dnsResults" class="results-list"></div>
      </div>
    `,
    init: () => {
      document.getElementById('dnsLookup').onclick = async () => {
        const domain = document.getElementById('dnsInput').value.trim();
        if (!domain) { alert('Enter a domain'); return; }

        const results = document.getElementById('dnsResults');
        results.innerHTML = '<div class="loading">Looking up DNS records...</div>';

        try {
          const res = await fetch(`${API_BASE}/dns.php?domain=${encodeURIComponent(domain)}`);
          const data = await res.json();

          if (data.error) {
            results.innerHTML = `<div class="error">${data.error}</div>`;
            return;
          }

          let html = '';
          for (const [type, records] of Object.entries(data)) {
            if (Array.isArray(records) && records.length > 0) {
              html += `<div class="dns-section"><h4>${type}</h4>`;
              records.forEach(r => {
                html += `<div class="dns-record">${typeof r === 'object' ? JSON.stringify(r) : r}</div>`;
              });
              html += '</div>';
            }
          }
          results.innerHTML = html || '<div class="no-results">No records found</div>';
        } catch (e) {
          results.innerHTML = `<div class="error">Error: ${e.message}</div>`;
        }
      };
    }
  },

  // ==================== WHOIS (API) ====================
  'whois': {
    title: 'WHOIS Lookup',
    render: () => `
      <div class="tool-section">
        <div class="form-group">
          <label>Domain Name</label>
          <input type="text" id="whoisInput" placeholder="example.com">
        </div>
        <button class="btn-primary" id="whoisLookup">WHOIS Lookup</button>
        <div class="form-group">
          <label>Results</label>
          <textarea id="whoisOutput" rows="10" readonly></textarea>
        </div>
      </div>
    `,
    init: () => {
      document.getElementById('whoisLookup').onclick = async () => {
        const domain = document.getElementById('whoisInput').value.trim();
        if (!domain) { alert('Enter a domain'); return; }

        document.getElementById('whoisOutput').value = 'Looking up WHOIS data...';

        try {
          const res = await fetch(`${API_BASE}/whois.php?domain=${encodeURIComponent(domain)}`);
          const data = await res.json();

          if (data.error) {
            document.getElementById('whoisOutput').value = 'Error: ' + data.error;
            return;
          }

          document.getElementById('whoisOutput').value = data.raw || JSON.stringify(data, null, 2);
        } catch (e) {
          document.getElementById('whoisOutput').value = 'Error: ' + e.message;
        }
      };
    }
  },

  // ==================== IP LOOKUP (API) ====================
  'ip-lookup': {
    title: 'IP Geolocation',
    render: () => `
      <div class="tool-section">
        <div class="form-group">
          <label>IP Address (leave empty for your IP)</label>
          <input type="text" id="ipInput" placeholder="8.8.8.8">
        </div>
        <button class="btn-primary" id="ipLookup">Lookup IP</button>
        <div id="ipResults" class="results-list"></div>
      </div>
    `,
    init: () => {
      document.getElementById('ipLookup').onclick = async () => {
        const ip = document.getElementById('ipInput').value.trim();
        const results = document.getElementById('ipResults');
        results.innerHTML = '<div class="loading">Looking up IP...</div>';

        try {
          const url = ip ? `${API_BASE}/ip-lookup.php?ip=${encodeURIComponent(ip)}` : `${API_BASE}/ip-lookup.php`;
          const res = await fetch(url);
          const data = await res.json();

          if (data.error) {
            results.innerHTML = `<div class="error">${data.error}</div>`;
            return;
          }

          results.innerHTML = `
            <div class="result-item"><label>IP</label><span>${data.ip}</span></div>
            <div class="result-item"><label>City</label><span>${data.city || 'N/A'}</span></div>
            <div class="result-item"><label>Region</label><span>${data.region || 'N/A'}</span></div>
            <div class="result-item"><label>Country</label><span>${data.country || 'N/A'}</span></div>
            <div class="result-item"><label>ISP</label><span>${data.isp || 'N/A'}</span></div>
            <div class="result-item"><label>Timezone</label><span>${data.timezone || 'N/A'}</span></div>
          `;
        } catch (e) {
          results.innerHTML = `<div class="error">Error: ${e.message}</div>`;
        }
      };
      document.getElementById('ipLookup').click();
    }
  },

  // ==================== HTTP HEADERS (API) ====================
  'headers': {
    title: 'HTTP Headers',
    render: () => `
      <div class="tool-section">
        <div class="form-group">
          <label>URL</label>
          <input type="text" id="headersInput" placeholder="https://example.com">
        </div>
        <button class="btn-primary" id="headersLookup">Get Headers</button>
        <div class="form-group">
          <label>Response Headers</label>
          <textarea id="headersOutput" rows="10" readonly></textarea>
        </div>
      </div>
    `,
    init: () => {
      document.getElementById('headersLookup').onclick = async () => {
        let url = document.getElementById('headersInput').value.trim();
        if (!url) { alert('Enter a URL'); return; }
        if (!url.startsWith('http')) url = 'https://' + url;

        document.getElementById('headersOutput').value = 'Fetching headers...';

        try {
          const res = await fetch(`${API_BASE}/headers.php?url=${encodeURIComponent(url)}`);
          const data = await res.json();

          if (data.error) {
            document.getElementById('headersOutput').value = 'Error: ' + data.error;
            return;
          }

          let output = '';
          for (const [key, value] of Object.entries(data.headers || data)) {
            output += `${key}: ${value}\n`;
          }
          document.getElementById('headersOutput').value = output || 'No headers returned';
        } catch (e) {
          document.getElementById('headersOutput').value = 'Error: ' + e.message;
        }
      };
    }
  },

  // ==================== NOTES ====================
  'notes': {
    title: 'My Notes',
    render: () => `
      <div class="tool-section">
        <div id="notesListView">
          <div style="display:flex;justify-content:space-between;align-items:center;margin-bottom:12px;">
            <span id="notesStatus" style="font-size:11px;color:var(--text-muted)"></span>
            <button class="btn-primary" id="newNoteBtn" style="padding:6px 12px;font-size:12px;">+ New Note</button>
          </div>
          <div id="notesList" style="max-height:280px;overflow-y:auto;"></div>
        </div>
        <div id="notesEditView" style="display:none;">
          <div style="display:flex;justify-content:space-between;align-items:center;margin-bottom:12px;">
            <button class="btn-secondary" id="backToListBtn" style="padding:6px 12px;font-size:12px;">Back</button>
            <button class="btn-primary" id="saveNoteBtn" style="padding:6px 12px;font-size:12px;">Save Note</button>
          </div>
          <div class="form-group">
            <label>Title</label>
            <input type="text" id="noteTitle" placeholder="Note title...">
          </div>
          <div class="form-group">
            <label>Content</label>
            <textarea id="noteContent" rows="8" placeholder="Write your note here..."></textarea>
          </div>
        </div>
        <p style="font-size:11px;color:var(--text-muted);margin-top:12px;">Notes sync with dashboard.</p>
      </div>
    `,
    init: () => {
      const listView = document.getElementById('notesListView');
      const editView = document.getElementById('notesEditView');
      const notesList = document.getElementById('notesList');
      const notesStatus = document.getElementById('notesStatus');
      const noteTitle = document.getElementById('noteTitle');
      const noteContent = document.getElementById('noteContent');
      let editingId = null;

      const getToken = () => new Promise(r => chrome.storage.local.get(['mnt_session'], res => r(res.mnt_session?.token)));

      const showList = () => {
        listView.style.display = 'block';
        editView.style.display = 'none';
        editingId = null;
      };

      const showEdit = (note = null) => {
        listView.style.display = 'none';
        editView.style.display = 'block';
        if (note) {
          editingId = note.id;
          noteTitle.value = note.title || '';
          noteContent.value = note.content || '';
        } else {
          editingId = null;
          noteTitle.value = '';
          noteContent.value = '';
        }
        noteTitle.focus();
      };

      const loadNotes = async () => {
        const token = await getToken();
        if (!token) {
          notesList.innerHTML = '<div style="text-align:center;padding:20px;color:var(--text-muted);">Login required</div>';
          return;
        }
        try {
          notesStatus.textContent = 'Loading...';
          // Send token as query param (more reliable) AND in header (for good measure)
          const res = await fetch(`${API_BASE}/notes.php?action=list&token=${encodeURIComponent(token)}`, {
            headers: { 'Authorization': `Bearer ${token}` }
          });
          const data = await res.json();
          if (data.success && data.notes) {
            if (data.notes.length === 0) {
              notesList.innerHTML = '<div style="text-align:center;padding:20px;color:var(--text-muted);">No notes yet.<br>Create your first note!</div>';
            } else {
              notesList.innerHTML = data.notes.map(note => `
                <div class="note-item" data-id="${note.id}" style="background:var(--bg-secondary);border:1px solid var(--border);border-radius:6px;padding:10px;margin-bottom:8px;cursor:pointer;">
                  <div style="font-weight:500;margin-bottom:4px;white-space:nowrap;overflow:hidden;text-overflow:ellipsis;">${escapeHtml(note.title || 'Untitled')}</div>
                  <div style="font-size:11px;color:var(--text-muted);white-space:nowrap;overflow:hidden;text-overflow:ellipsis;">${escapeHtml((note.content || '').substring(0, 60))}</div>
                  <div style="font-size:10px;color:var(--text-muted);margin-top:4px;">${new Date(note.created_at).toLocaleDateString()}</div>
                </div>
              `).join('');
              document.querySelectorAll('.note-item').forEach(item => {
                item.onclick = () => {
                  const id = item.dataset.id;
                  const note = data.notes.find(n => n.id == id);
                  if (note) showEdit(note);
                };
              });
            }
            notesStatus.textContent = `${data.notes.length} note${data.notes.length !== 1 ? 's' : ''}`;
          }
        } catch (e) {
          notesList.innerHTML = '<div style="text-align:center;padding:20px;color:#ef4444;">Failed to load notes</div>';
          notesStatus.textContent = 'Error';
        }
      };

      const saveNote = async () => {
        const token = await getToken();
        if (!token) return alert('Login required');
        const title = noteTitle.value.trim() || 'Untitled';
        const content = noteContent.value;
        try {
          // Send token as query param (more reliable) AND in header
          const res = await fetch(`${API_BASE}/notes.php?action=create&token=${encodeURIComponent(token)}`, {
            method: 'POST',
            headers: {
              'Content-Type': 'application/json',
              'Authorization': `Bearer ${token}`
            },
            body: JSON.stringify({ title, content })
          });
          const data = await res.json();
          if (data.success) {
            showList();
            loadNotes();
          } else {
            alert('Failed to save note');
          }
        } catch (e) {
          alert('Error saving note');
        }
      };

      document.getElementById('newNoteBtn').onclick = () => showEdit();
      document.getElementById('backToListBtn').onclick = showList;
      document.getElementById('saveNoteBtn').onclick = saveNote;

      loadNotes();
    }
  },

  // ==================== PASSWORD STRENGTH ====================
  'password-check': {
    title: 'Password Strength Checker',
    render: () => `
      <div class="tool-section">
        <div class="form-group">
          <label>Enter Password to Check</label>
          <input type="text" id="passCheckInput" placeholder="Enter password...">
        </div>
        <div id="strengthResult" class="results-list"></div>
      </div>
    `,
    init: () => {
      const input = document.getElementById('passCheckInput');
      const result = document.getElementById('strengthResult');

      input.oninput = () => {
        const pass = input.value;
        if (!pass) { result.innerHTML = ''; return; }

        let score = 0;
        const checks = {
          length: pass.length >= 8,
          length12: pass.length >= 12,
          upper: /[A-Z]/.test(pass),
          lower: /[a-z]/.test(pass),
          number: /[0-9]/.test(pass),
          special: /[^A-Za-z0-9]/.test(pass),
          noCommon: !/(password|123456|qwerty|admin)/i.test(pass)
        };

        if (checks.length) score += 1;
        if (checks.length12) score += 1;
        if (checks.upper) score += 1;
        if (checks.lower) score += 1;
        if (checks.number) score += 1;
        if (checks.special) score += 2;
        if (checks.noCommon) score += 1;

        const strength = score >= 7 ? 'Strong' : score >= 5 ? 'Good' : score >= 3 ? 'Fair' : 'Weak';
        const color = score >= 7 ? '#4ade80' : score >= 5 ? '#6eb3ff' : score >= 3 ? '#fbbf24' : '#ef4444';

        result.innerHTML = `
          <div class="stat-item" style="text-align:center;padding:1rem;margin-bottom:1rem;">
            <span class="stat-value" style="color:${color}">${strength}</span>
            <span class="stat-label">Score: ${score}/8</span>
          </div>
          <div class="result-item"><label>Length 8+</label><span>${checks.length ? '✓' : '✗'}</span></div>
          <div class="result-item"><label>Length 12+</label><span>${checks.length12 ? '✓' : '✗'}</span></div>
          <div class="result-item"><label>Uppercase</label><span>${checks.upper ? '✓' : '✗'}</span></div>
          <div class="result-item"><label>Lowercase</label><span>${checks.lower ? '✓' : '✗'}</span></div>
          <div class="result-item"><label>Numbers</label><span>${checks.number ? '✓' : '✗'}</span></div>
          <div class="result-item"><label>Special Chars</label><span>${checks.special ? '✓' : '✗'}</span></div>
          <div class="result-item"><label>Not Common</label><span>${checks.noCommon ? '✓' : '✗'}</span></div>
        `;
      };
    }
  },

  // ==================== JWT DECODER ====================
  'jwt-decoder': {
    title: 'JWT Decoder',
    render: () => `
      <div class="tool-section">
        <div class="form-group">
          <label>JWT Token</label>
          <textarea id="jwtInput" rows="4" placeholder="eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9..."></textarea>
        </div>
        <button class="btn-primary" id="decodeJwt">Decode JWT</button>
        <div id="jwtResult" class="results-list"></div>
      </div>
    `,
    init: () => {
      document.getElementById('decodeJwt').onclick = () => {
        const jwt = document.getElementById('jwtInput').value.trim();
        const result = document.getElementById('jwtResult');

        if (!jwt) { result.innerHTML = '<div class="error">Enter a JWT token</div>'; return; }

        try {
          const parts = jwt.split('.');
          if (parts.length !== 3) throw new Error('Invalid JWT format');

          const header = JSON.parse(atob(parts[0].replace(/-/g, '+').replace(/_/g, '/')));
          const payload = JSON.parse(atob(parts[1].replace(/-/g, '+').replace(/_/g, '/')));

          let expInfo = '';
          if (payload.exp) {
            const expDate = new Date(payload.exp * 1000);
            const isExpired = expDate < new Date();
            expInfo = `<div class="result-item"><label>Expires</label><span style="color:${isExpired ? '#ef4444' : '#4ade80'}">${expDate.toLocaleString()} ${isExpired ? '(EXPIRED)' : ''}</span></div>`;
          }

          result.innerHTML = `
            <div class="dns-section"><h4>Header</h4><div class="dns-record">${JSON.stringify(header, null, 2)}</div></div>
            <div class="dns-section"><h4>Payload</h4><div class="dns-record">${JSON.stringify(payload, null, 2)}</div></div>
            ${expInfo}
          `;
        } catch (e) {
          result.innerHTML = '<div class="error">Invalid JWT: ' + e.message + '</div>';
        }
      };
    }
  },

  // ==================== QR CODE GENERATOR ====================
  'qr-gen': {
    title: 'QR Code Generator',
    render: () => `
      <div class="tool-section">
        <div class="form-group">
          <label>Text or URL</label>
          <input type="text" id="qrInput" placeholder="https://example.com or any text">
        </div>
        <div class="form-group">
          <label>Size</label>
          <select id="qrSize" style="width:100%;padding:10px;background:var(--bg-secondary);border:1px solid var(--border);border-radius:6px;color:var(--text);">
            <option value="150">Small (150x150)</option>
            <option value="200" selected>Medium (200x200)</option>
            <option value="300">Large (300x300)</option>
          </select>
        </div>
        <button class="btn-primary" id="generateQr">Generate QR Code</button>
        <div id="qrResult" style="text-align:center;margin-top:1rem;"></div>
      </div>
    `,
    init: () => {
      document.getElementById('generateQr').onclick = () => {
        const text = document.getElementById('qrInput').value.trim();
        const size = document.getElementById('qrSize').value;
        const result = document.getElementById('qrResult');

        if (!text) { result.innerHTML = '<div class="error">Enter text or URL</div>'; return; }

        // Using Google Charts API for QR generation
        const qrUrl = `https://chart.googleapis.com/chart?chs=${size}x${size}&cht=qr&chl=${encodeURIComponent(text)}&choe=UTF-8`;
        result.innerHTML = `<img src="${qrUrl}" alt="QR Code" style="border-radius:8px;background:#fff;padding:8px;"><br><a href="${qrUrl}" download="qrcode.png" class="btn-secondary" style="display:inline-block;margin-top:1rem;">Download</a>`;
      };
    }
  },

  // ==================== MARKDOWN PREVIEW ====================
  'markdown': {
    title: 'Markdown Preview',
    render: () => `
      <div class="tool-section">
        <div class="form-group">
          <label>Markdown Input</label>
          <textarea id="mdInput" rows="6" placeholder="# Heading\n\n**Bold** and *italic*\n\n- List item\n- Another item"></textarea>
        </div>
        <div class="form-group">
          <label>Preview</label>
          <div id="mdPreview" style="background:var(--bg-secondary);border:1px solid var(--border);border-radius:6px;padding:12px;min-height:100px;"></div>
        </div>
      </div>
    `,
    init: () => {
      const input = document.getElementById('mdInput');
      const preview = document.getElementById('mdPreview');

      const renderMd = () => {
        let md = input.value;
        // Simple markdown parser
        md = md.replace(/^### (.*$)/gm, '<h4>$1</h4>');
        md = md.replace(/^## (.*$)/gm, '<h3>$1</h3>');
        md = md.replace(/^# (.*$)/gm, '<h2>$1</h2>');
        md = md.replace(/\*\*(.+?)\*\*/g, '<strong>$1</strong>');
        md = md.replace(/\*(.+?)\*/g, '<em>$1</em>');
        md = md.replace(/`(.+?)`/g, '<code style="background:var(--bg-tertiary);padding:2px 6px;border-radius:3px;">$1</code>');
        md = md.replace(/^- (.*)$/gm, '<li>$1</li>');
        md = md.replace(/(<li>.*<\/li>)/s, '<ul>$1</ul>');
        md = md.replace(/\n\n/g, '<br><br>');
        md = md.replace(/\[(.+?)\]\((.+?)\)/g, '<a href="$2" target="_blank">$1</a>');
        preview.innerHTML = md || '<span style="color:var(--text-muted)">Preview will appear here...</span>';
      };

      input.oninput = renderMd;
      renderMd();
    }
  },

  // ==================== DIFF COMPARE ====================
  'diff': {
    title: 'Text Diff Compare',
    render: () => `
      <div class="tool-section">
        <div class="form-group">
          <label>Original Text</label>
          <textarea id="diffOriginal" rows="4" placeholder="Original text..."></textarea>
        </div>
        <div class="form-group">
          <label>Modified Text</label>
          <textarea id="diffModified" rows="4" placeholder="Modified text..."></textarea>
        </div>
        <button class="btn-primary" id="compareDiff">Compare</button>
        <div class="form-group">
          <label>Differences</label>
          <div id="diffResult" style="background:var(--bg-secondary);border:1px solid var(--border);border-radius:6px;padding:12px;font-family:monospace;font-size:12px;white-space:pre-wrap;max-height:200px;overflow-y:auto;"></div>
        </div>
      </div>
    `,
    init: () => {
      document.getElementById('compareDiff').onclick = () => {
        const orig = document.getElementById('diffOriginal').value.split('\n');
        const mod = document.getElementById('diffModified').value.split('\n');
        const result = document.getElementById('diffResult');

        let html = '';
        const maxLines = Math.max(orig.length, mod.length);

        for (let i = 0; i < maxLines; i++) {
          const o = orig[i] || '';
          const m = mod[i] || '';

          if (o === m) {
            html += `<div style="color:var(--text-secondary)">${escapeHtml(o) || '&nbsp;'}</div>`;
          } else if (!o && m) {
            html += `<div style="color:#4ade80;background:rgba(74,222,128,0.1)">+ ${escapeHtml(m)}</div>`;
          } else if (o && !m) {
            html += `<div style="color:#ef4444;background:rgba(239,68,68,0.1)">- ${escapeHtml(o)}</div>`;
          } else {
            html += `<div style="color:#ef4444;background:rgba(239,68,68,0.1)">- ${escapeHtml(o)}</div>`;
            html += `<div style="color:#4ade80;background:rgba(74,222,128,0.1)">+ ${escapeHtml(m)}</div>`;
          }
        }

        result.innerHTML = html || '<span style="color:var(--text-muted)">No differences</span>';
      };
    }
  },

  // ==================== NUMBER BASE CONVERTER ====================
  'base-convert': {
    title: 'Number Base Converter',
    render: () => `
      <div class="tool-section">
        <div class="form-group">
          <label>Input Number</label>
          <input type="text" id="baseInput" placeholder="Enter a number">
        </div>
        <div class="form-group">
          <label>Input Base</label>
          <select id="baseFrom" style="width:100%;padding:10px;background:var(--bg-secondary);border:1px solid var(--border);border-radius:6px;color:var(--text);">
            <option value="10" selected>Decimal (10)</option>
            <option value="2">Binary (2)</option>
            <option value="8">Octal (8)</option>
            <option value="16">Hexadecimal (16)</option>
          </select>
        </div>
        <button class="btn-primary" id="convertBase">Convert</button>
        <div class="results-list" id="baseResults"></div>
      </div>
    `,
    init: () => {
      document.getElementById('convertBase').onclick = () => {
        const input = document.getElementById('baseInput').value.trim();
        const fromBase = parseInt(document.getElementById('baseFrom').value);
        const results = document.getElementById('baseResults');

        if (!input) { results.innerHTML = '<div class="error">Enter a number</div>'; return; }

        try {
          const decimal = parseInt(input, fromBase);
          if (isNaN(decimal)) throw new Error('Invalid number for selected base');

          results.innerHTML = `
            <div class="result-item"><label>Decimal</label><div class="result-box"><input type="text" value="${decimal}" readonly><button class="btn-copy" data-value="${decimal}">Copy</button></div></div>
            <div class="result-item"><label>Binary</label><div class="result-box"><input type="text" value="${decimal.toString(2)}" readonly><button class="btn-copy" data-value="${decimal.toString(2)}">Copy</button></div></div>
            <div class="result-item"><label>Octal</label><div class="result-box"><input type="text" value="${decimal.toString(8)}" readonly><button class="btn-copy" data-value="${decimal.toString(8)}">Copy</button></div></div>
            <div class="result-item"><label>Hexadecimal</label><div class="result-box"><input type="text" value="${decimal.toString(16).toUpperCase()}" readonly><button class="btn-copy" data-value="${decimal.toString(16).toUpperCase()}">Copy</button></div></div>
          `;
          setupCopyButtons();
        } catch (e) {
          results.innerHTML = '<div class="error">' + e.message + '</div>';
        }
      };
    }
  },

  // ==================== SUBNET CALCULATOR ====================
  'subnet': {
    title: 'Subnet Calculator',
    render: () => `
      <div class="tool-section">
        <div class="form-group">
          <label>IP Address</label>
          <input type="text" id="subnetIp" placeholder="192.168.1.0">
        </div>
        <div class="form-group">
          <label>CIDR / Subnet Mask</label>
          <select id="subnetCidr" style="width:100%;padding:10px;background:var(--bg-secondary);border:1px solid var(--border);border-radius:6px;color:var(--text);">
            <option value="8">/8 - 255.0.0.0 (16M hosts)</option>
            <option value="16">/16 - 255.255.0.0 (65K hosts)</option>
            <option value="20">/20 - 255.255.240.0 (4K hosts)</option>
            <option value="24" selected>/24 - 255.255.255.0 (254 hosts)</option>
            <option value="25">/25 - 255.255.255.128 (126 hosts)</option>
            <option value="26">/26 - 255.255.255.192 (62 hosts)</option>
            <option value="27">/27 - 255.255.255.224 (30 hosts)</option>
            <option value="28">/28 - 255.255.255.240 (14 hosts)</option>
            <option value="29">/29 - 255.255.255.248 (6 hosts)</option>
            <option value="30">/30 - 255.255.255.252 (2 hosts)</option>
          </select>
        </div>
        <button class="btn-primary" id="calcSubnet">Calculate</button>
        <div class="results-list" id="subnetResults"></div>
      </div>
    `,
    init: () => {
      document.getElementById('calcSubnet').onclick = () => {
        const ip = document.getElementById('subnetIp').value.trim();
        const cidr = parseInt(document.getElementById('subnetCidr').value);
        const results = document.getElementById('subnetResults');

        if (!ip.match(/^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$/)) {
          results.innerHTML = '<div class="error">Enter a valid IP address</div>';
          return;
        }

        const parts = ip.split('.').map(Number);
        const ipNum = (parts[0] << 24) + (parts[1] << 16) + (parts[2] << 8) + parts[3];
        const mask = ~((1 << (32 - cidr)) - 1);
        const network = ipNum & mask;
        const broadcast = network | ~mask;
        const firstHost = network + 1;
        const lastHost = broadcast - 1;
        const hostCount = Math.pow(2, 32 - cidr) - 2;

        const numToIp = (num) => [(num >>> 24) & 255, (num >>> 16) & 255, (num >>> 8) & 255, num & 255].join('.');
        const maskIp = numToIp(mask >>> 0);

        results.innerHTML = `
          <div class="result-item"><label>Network Address</label><span>${numToIp(network >>> 0)}</span></div>
          <div class="result-item"><label>Subnet Mask</label><span>${maskIp}</span></div>
          <div class="result-item"><label>Broadcast Address</label><span>${numToIp(broadcast >>> 0)}</span></div>
          <div class="result-item"><label>First Host</label><span>${numToIp(firstHost >>> 0)}</span></div>
          <div class="result-item"><label>Last Host</label><span>${numToIp(lastHost >>> 0)}</span></div>
          <div class="result-item"><label>Usable Hosts</label><span>${hostCount.toLocaleString()}</span></div>
        `;
      };
    }
  },

  // ==================== ASCII TABLE ====================
  'ascii': {
    title: 'ASCII Table',
    render: () => `
      <div class="tool-section">
        <div class="form-group">
          <label>Character or Code</label>
          <input type="text" id="asciiInput" placeholder="A or 65">
        </div>
        <button class="btn-primary" id="lookupAscii">Lookup</button>
        <div id="asciiResult" class="results-list"></div>
        <hr>
        <div style="max-height:200px;overflow-y:auto;">
          <table style="width:100%;font-size:11px;border-collapse:collapse;">
            <tr style="background:var(--bg-tertiary)"><th style="padding:4px">Dec</th><th>Hex</th><th>Char</th><th style="border-left:1px solid var(--border);padding-left:8px">Dec</th><th>Hex</th><th>Char</th></tr>
            ${(() => {
              let rows = '';
              for (let i = 32; i < 80; i++) {
                const j = i + 48;
                rows += `<tr><td style="padding:4px">${i}</td><td>${i.toString(16).toUpperCase()}</td><td>${String.fromCharCode(i)}</td><td style="border-left:1px solid var(--border);padding-left:8px">${j}</td><td>${j.toString(16).toUpperCase()}</td><td>${j < 127 ? String.fromCharCode(j) : ''}</td></tr>`;
              }
              return rows;
            })()}
          </table>
        </div>
      </div>
    `,
    init: () => {
      document.getElementById('lookupAscii').onclick = () => {
        const input = document.getElementById('asciiInput').value.trim();
        const result = document.getElementById('asciiResult');

        if (!input) { result.innerHTML = ''; return; }

        let char, code;
        if (/^\d+$/.test(input)) {
          code = parseInt(input);
          char = String.fromCharCode(code);
        } else {
          char = input[0];
          code = char.charCodeAt(0);
        }

        result.innerHTML = `
          <div class="result-item"><label>Character</label><span style="font-size:24px">${char}</span></div>
          <div class="result-item"><label>Decimal</label><span>${code}</span></div>
          <div class="result-item"><label>Hexadecimal</label><span>0x${code.toString(16).toUpperCase()}</span></div>
          <div class="result-item"><label>Binary</label><span>${code.toString(2).padStart(8, '0')}</span></div>
          <div class="result-item"><label>Octal</label><span>${code.toString(8)}</span></div>
        `;
      };
    }
  },

  // ==================== SSL CHECKER (API) ====================
  'ssl-check': {
    title: 'SSL Certificate Checker',
    render: () => `
      <div class="tool-section">
        <div class="form-group">
          <label>Domain Name</label>
          <input type="text" id="sslInput" placeholder="example.com">
        </div>
        <button class="btn-primary" id="checkSsl">Check SSL</button>
        <div id="sslResults" class="results-list"></div>
      </div>
    `,
    init: () => {
      document.getElementById('checkSsl').onclick = async () => {
        const domain = document.getElementById('sslInput').value.trim();
        if (!domain) { alert('Enter a domain'); return; }

        const results = document.getElementById('sslResults');
        results.innerHTML = '<div class="loading">Checking SSL certificate...</div>';

        try {
          const res = await fetch(`${API_BASE}/ssl-check.php?domain=${encodeURIComponent(domain)}`);
          const data = await res.json();

          if (data.error) {
            results.innerHTML = `<div class="error">${data.error}</div>`;
            return;
          }

          const daysLeft = data.days_left || 0;
          const color = daysLeft > 30 ? '#4ade80' : daysLeft > 7 ? '#fbbf24' : '#ef4444';

          results.innerHTML = `
            <div class="stat-item" style="text-align:center;padding:1rem;margin-bottom:1rem;">
              <span class="stat-value" style="color:${color}">${daysLeft}</span>
              <span class="stat-label">Days Until Expiry</span>
            </div>
            <div class="result-item"><label>Issuer</label><span>${data.issuer || 'N/A'}</span></div>
            <div class="result-item"><label>Valid From</label><span>${data.valid_from || 'N/A'}</span></div>
            <div class="result-item"><label>Valid To</label><span>${data.valid_to || 'N/A'}</span></div>
            <div class="result-item"><label>Subject</label><span>${data.subject || 'N/A'}</span></div>
          `;
        } catch (e) {
          results.innerHTML = `<div class="error">Error: ${e.message}</div>`;
        }
      };
    }
  },

  // ==================== PING (API) ====================
  'ping': {
    title: 'Ping Test',
    render: () => `
      <div class="tool-section">
        <div class="form-group">
          <label>Host or IP</label>
          <input type="text" id="pingInput" placeholder="google.com or 8.8.8.8">
        </div>
        <button class="btn-primary" id="runPing">Ping</button>
        <div id="pingResults" class="results-list"></div>
      </div>
    `,
    init: () => {
      document.getElementById('runPing').onclick = async () => {
        const host = document.getElementById('pingInput').value.trim();
        if (!host) { alert('Enter a host'); return; }

        const results = document.getElementById('pingResults');
        results.innerHTML = '<div class="loading">Pinging host...</div>';

        try {
          const res = await fetch(`${API_BASE}/ping.php?host=${encodeURIComponent(host)}`);
          const data = await res.json();

          if (data.error) {
            results.innerHTML = `<div class="error">${data.error}</div>`;
            return;
          }

          const status = data.alive ? 'Online' : 'Offline';
          const color = data.alive ? '#4ade80' : '#ef4444';

          results.innerHTML = `
            <div class="stat-item" style="text-align:center;padding:1rem;margin-bottom:1rem;">
              <span class="stat-value" style="color:${color}">${status}</span>
              <span class="stat-label">${data.ip || host}</span>
            </div>
            ${data.time ? `<div class="result-item"><label>Response Time</label><span>${data.time}ms</span></div>` : ''}
            ${data.ttl ? `<div class="result-item"><label>TTL</label><span>${data.ttl}</span></div>` : ''}
          `;
        } catch (e) {
          results.innerHTML = `<div class="error">Error: ${e.message}</div>`;
        }
      };
    }
  },

  // ==================== REVERSE DNS (API) ====================
  'reverse-dns': {
    title: 'Reverse DNS Lookup',
    render: () => `
      <div class="tool-section">
        <div class="form-group">
          <label>IP Address</label>
          <input type="text" id="rdnsInput" placeholder="8.8.8.8">
        </div>
        <button class="btn-primary" id="lookupRdns">Lookup</button>
        <div id="rdnsResults" class="results-list"></div>
      </div>
    `,
    init: () => {
      document.getElementById('lookupRdns').onclick = async () => {
        const ip = document.getElementById('rdnsInput').value.trim();
        if (!ip) { alert('Enter an IP address'); return; }

        const results = document.getElementById('rdnsResults');
        results.innerHTML = '<div class="loading">Looking up...</div>';

        try {
          const res = await fetch(`${API_BASE}/reverse-dns.php?ip=${encodeURIComponent(ip)}`);
          const data = await res.json();

          if (data.error) {
            results.innerHTML = `<div class="error">${data.error}</div>`;
            return;
          }

          results.innerHTML = `
            <div class="result-item"><label>IP Address</label><span>${data.ip || ip}</span></div>
            <div class="result-item"><label>Hostname</label><span>${data.hostname || 'No PTR record found'}</span></div>
          `;
        } catch (e) {
          results.innerHTML = `<div class="error">Error: ${e.message}</div>`;
        }
      };
    }
  },

  // ==================== PORT SCANNER (API) ====================
  'port-scan': {
    title: 'Port Scanner',
    render: () => `
      <div class="tool-section">
        <div class="form-group">
          <label>Host</label>
          <input type="text" id="portHost" placeholder="example.com">
        </div>
        <div class="form-group">
          <label>Common Ports</label>
          <div class="checkbox-group">
            <label><input type="checkbox" value="21" checked> FTP (21)</label>
            <label><input type="checkbox" value="22" checked> SSH (22)</label>
            <label><input type="checkbox" value="80" checked> HTTP (80)</label>
            <label><input type="checkbox" value="443" checked> HTTPS (443)</label>
            <label><input type="checkbox" value="3306"> MySQL (3306)</label>
            <label><input type="checkbox" value="3389"> RDP (3389)</label>
          </div>
        </div>
        <button class="btn-primary" id="scanPorts">Scan Ports</button>
        <div id="portResults" class="results-list"></div>
      </div>
    `,
    init: () => {
      document.getElementById('scanPorts').onclick = async () => {
        const host = document.getElementById('portHost').value.trim();
        const checkboxes = document.querySelectorAll('#toolContent .checkbox-group input:checked');
        const ports = Array.from(checkboxes).map(cb => cb.value).join(',');

        if (!host) { alert('Enter a host'); return; }
        if (!ports) { alert('Select at least one port'); return; }

        const results = document.getElementById('portResults');
        results.innerHTML = '<div class="loading">Scanning ports...</div>';

        try {
          const res = await fetch(`${API_BASE}/port-scan.php?host=${encodeURIComponent(host)}&ports=${ports}`);
          const data = await res.json();

          if (data.error) {
            results.innerHTML = `<div class="error">${data.error}</div>`;
            return;
          }

          let html = '';
          for (const [port, status] of Object.entries(data.ports || {})) {
            const isOpen = status === 'open';
            html += `<div class="result-item"><label>Port ${port}</label><span style="color:${isOpen ? '#4ade80' : '#ef4444'}">${status.toUpperCase()}</span></div>`;
          }

          results.innerHTML = html || '<div class="no-results">No results</div>';
        } catch (e) {
          results.innerHTML = `<div class="error">Error: ${e.message}</div>`;
        }
      };
    }
  },

  // ==================== SITE SUBMIT ====================
  'site-submit': {
    title: 'Site Submit',
    render: () => `
      <div class="tool-section">
        <p style="color:var(--text-secondary);margin-bottom:1rem;">Submit your site to search engines, directories, and social platforms.</p>
        <div class="submit-features">
          <div class="feature-item" style="padding:0.75rem;background:var(--bg-secondary);border-radius:6px;margin-bottom:0.5rem;">
            <span style="color:#4ade80;">&#10003;</span> Ping Google, Bing, Yandex, IndexNow
          </div>
          <div class="feature-item" style="padding:0.75rem;background:var(--bg-secondary);border-radius:6px;margin-bottom:0.5rem;">
            <span style="color:#4ade80;">&#10003;</span> Submit to directories & platforms
          </div>
          <div class="feature-item" style="padding:0.75rem;background:var(--bg-secondary);border-radius:6px;margin-bottom:0.5rem;">
            <span style="color:#4ade80;">&#10003;</span> Social bookmarking shortcuts
          </div>
          <div class="feature-item" style="padding:0.75rem;background:var(--bg-secondary);border-radius:6px;margin-bottom:1rem;">
            <span style="color:#4ade80;">&#10003;</span> Webmaster tools quick links
          </div>
        </div>
        <button class="btn-primary" id="openSiteSubmit" style="width:100%;">Open Site Submit Tool</button>
        <p style="color:var(--text-muted);font-size:0.85rem;margin-top:0.75rem;text-align:center;">Opens on MyNetworkTools website</p>
      </div>
    `,
    init: () => {
      document.getElementById('openSiteSubmit').onclick = () => {
        window.open('https://mynetworktools.com/tools/site-submit.php', '_blank');
      };
    }
  }
};

// ==================== HELPER FUNCTIONS ====================

function simpleMD5(string) {
  function md5cycle(x, k) {
    var a = x[0], b = x[1], c = x[2], d = x[3];
    a = ff(a, b, c, d, k[0], 7, -680876936); d = ff(d, a, b, c, k[1], 12, -389564586);
    c = ff(c, d, a, b, k[2], 17, 606105819); b = ff(b, c, d, a, k[3], 22, -1044525330);
    a = ff(a, b, c, d, k[4], 7, -176418897); d = ff(d, a, b, c, k[5], 12, 1200080426);
    c = ff(c, d, a, b, k[6], 17, -1473231341); b = ff(b, c, d, a, k[7], 22, -45705983);
    a = ff(a, b, c, d, k[8], 7, 1770035416); d = ff(d, a, b, c, k[9], 12, -1958414417);
    c = ff(c, d, a, b, k[10], 17, -42063); b = ff(b, c, d, a, k[11], 22, -1990404162);
    a = ff(a, b, c, d, k[12], 7, 1804603682); d = ff(d, a, b, c, k[13], 12, -40341101);
    c = ff(c, d, a, b, k[14], 17, -1502002290); b = ff(b, c, d, a, k[15], 22, 1236535329);
    a = gg(a, b, c, d, k[1], 5, -165796510); d = gg(d, a, b, c, k[6], 9, -1069501632);
    c = gg(c, d, a, b, k[11], 14, 643717713); b = gg(b, c, d, a, k[0], 20, -373897302);
    a = gg(a, b, c, d, k[5], 5, -701558691); d = gg(d, a, b, c, k[10], 9, 38016083);
    c = gg(c, d, a, b, k[15], 14, -660478335); b = gg(b, c, d, a, k[4], 20, -405537848);
    a = gg(a, b, c, d, k[9], 5, 568446438); d = gg(d, a, b, c, k[14], 9, -1019803690);
    c = gg(c, d, a, b, k[3], 14, -187363961); b = gg(b, c, d, a, k[8], 20, 1163531501);
    a = gg(a, b, c, d, k[13], 5, -1444681467); d = gg(d, a, b, c, k[2], 9, -51403784);
    c = gg(c, d, a, b, k[7], 14, 1735328473); b = gg(b, c, d, a, k[12], 20, -1926607734);
    a = hh(a, b, c, d, k[5], 4, -378558); d = hh(d, a, b, c, k[8], 11, -2022574463);
    c = hh(c, d, a, b, k[11], 16, 1839030562); b = hh(b, c, d, a, k[14], 23, -35309556);
    a = hh(a, b, c, d, k[1], 4, -1530992060); d = hh(d, a, b, c, k[4], 11, 1272893353);
    c = hh(c, d, a, b, k[7], 16, -155497632); b = hh(b, c, d, a, k[10], 23, -1094730640);
    a = hh(a, b, c, d, k[13], 4, 681279174); d = hh(d, a, b, c, k[0], 11, -358537222);
    c = hh(c, d, a, b, k[3], 16, -722521979); b = hh(b, c, d, a, k[6], 23, 76029189);
    a = hh(a, b, c, d, k[9], 4, -640364487); d = hh(d, a, b, c, k[12], 11, -421815835);
    c = hh(c, d, a, b, k[15], 16, 530742520); b = hh(b, c, d, a, k[2], 23, -995338651);
    a = ii(a, b, c, d, k[0], 6, -198630844); d = ii(d, a, b, c, k[7], 10, 1126891415);
    c = ii(c, d, a, b, k[14], 15, -1416354905); b = ii(b, c, d, a, k[5], 21, -57434055);
    a = ii(a, b, c, d, k[12], 6, 1700485571); d = ii(d, a, b, c, k[3], 10, -1894986606);
    c = ii(c, d, a, b, k[10], 15, -1051523); b = ii(b, c, d, a, k[1], 21, -2054922799);
    a = ii(a, b, c, d, k[8], 6, 1873313359); d = ii(d, a, b, c, k[15], 10, -30611744);
    c = ii(c, d, a, b, k[6], 15, -1560198380); b = ii(b, c, d, a, k[13], 21, 1309151649);
    a = ii(a, b, c, d, k[4], 6, -145523070); d = ii(d, a, b, c, k[11], 10, -1120210379);
    c = ii(c, d, a, b, k[2], 15, 718787259); b = ii(b, c, d, a, k[9], 21, -343485551);
    x[0] = add32(a, x[0]); x[1] = add32(b, x[1]); x[2] = add32(c, x[2]); x[3] = add32(d, x[3]);
  }
  function cmn(q, a, b, x, s, t) { a = add32(add32(a, q), add32(x, t)); return add32((a << s) | (a >>> (32 - s)), b); }
  function ff(a, b, c, d, x, s, t) { return cmn((b & c) | ((~b) & d), a, b, x, s, t); }
  function gg(a, b, c, d, x, s, t) { return cmn((b & d) | (c & (~d)), a, b, x, s, t); }
  function hh(a, b, c, d, x, s, t) { return cmn(b ^ c ^ d, a, b, x, s, t); }
  function ii(a, b, c, d, x, s, t) { return cmn(c ^ (b | (~d)), a, b, x, s, t); }
  function md51(s) {
    var n = s.length, state = [1732584193, -271733879, -1732584194, 271733878], i;
    for (i = 64; i <= s.length; i += 64) md5cycle(state, md5blk(s.substring(i - 64, i)));
    s = s.substring(i - 64); var tail = [0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0];
    for (i = 0; i < s.length; i++) tail[i >> 2] |= s.charCodeAt(i) << ((i % 4) << 3);
    tail[i >> 2] |= 0x80 << ((i % 4) << 3);
    if (i > 55) { md5cycle(state, tail); for (i = 0; i < 16; i++) tail[i] = 0; }
    tail[14] = n * 8; md5cycle(state, tail); return state;
  }
  function md5blk(s) { var md5blks = [], i; for (i = 0; i < 64; i += 4) md5blks[i >> 2] = s.charCodeAt(i) + (s.charCodeAt(i+1) << 8) + (s.charCodeAt(i+2) << 16) + (s.charCodeAt(i+3) << 24); return md5blks; }
  var hex_chr = '0123456789abcdef'.split('');
  function rhex(n) { var s = '', j = 0; for (; j < 4; j++) s += hex_chr[(n >> (j * 8 + 4)) & 0x0F] + hex_chr[(n >> (j * 8)) & 0x0F]; return s; }
  function hex(x) { for (var i = 0; i < x.length; i++) x[i] = rhex(x[i]); return x.join(''); }
  function add32(a, b) { return (a + b) & 0xFFFFFFFF; }
  return hex(md51(string));
}

async function encryptAES(text, password) {
  const enc = new TextEncoder();
  const keyMaterial = await crypto.subtle.importKey('raw', enc.encode(password), 'PBKDF2', false, ['deriveBits', 'deriveKey']);
  const salt = crypto.getRandomValues(new Uint8Array(16));
  const key = await crypto.subtle.deriveKey({ name: 'PBKDF2', salt, iterations: 100000, hash: 'SHA-256' }, keyMaterial, { name: 'AES-GCM', length: 256 }, false, ['encrypt']);
  const iv = crypto.getRandomValues(new Uint8Array(12));
  const encrypted = await crypto.subtle.encrypt({ name: 'AES-GCM', iv }, key, enc.encode(text));
  const combined = new Uint8Array(salt.length + iv.length + encrypted.byteLength);
  combined.set(salt, 0);
  combined.set(iv, salt.length);
  combined.set(new Uint8Array(encrypted), salt.length + iv.length);
  return btoa(String.fromCharCode(...combined));
}

async function decryptAES(encryptedBase64, password) {
  const enc = new TextEncoder();
  const dec = new TextDecoder();
  const combined = Uint8Array.from(atob(encryptedBase64), c => c.charCodeAt(0));
  const salt = combined.slice(0, 16);
  const iv = combined.slice(16, 28);
  const data = combined.slice(28);
  const keyMaterial = await crypto.subtle.importKey('raw', enc.encode(password), 'PBKDF2', false, ['deriveBits', 'deriveKey']);
  const key = await crypto.subtle.deriveKey({ name: 'PBKDF2', salt, iterations: 100000, hash: 'SHA-256' }, keyMaterial, { name: 'AES-GCM', length: 256 }, false, ['decrypt']);
  const decrypted = await crypto.subtle.decrypt({ name: 'AES-GCM', iv }, key, data);
  return dec.decode(decrypted);
}

function escapeHtml(text) {
  const div = document.createElement('div');
  div.textContent = text;
  return div.innerHTML;
}

function setupCopyButtons() {
  document.querySelectorAll('.btn-copy').forEach(btn => {
    btn.onclick = () => {
      const target = btn.dataset.copy ? document.getElementById(btn.dataset.copy) : null;
      const text = btn.dataset.value || (target ? target.value : '');
      navigator.clipboard.writeText(text).then(() => {
        const orig = btn.textContent;
        btn.textContent = 'Copied!';
        setTimeout(() => btn.textContent = orig, 1500);
      });
    };
  });
}
