// MyNetworkTools Chrome Extension - Main Controller
// API_BASE is defined in tools.js
const CURRENT_VERSION = '2.1.7';

document.addEventListener('DOMContentLoaded', function() {
  const loginScreen = document.getElementById('loginScreen');
  const toolsScreen = document.getElementById('toolsScreen');
  const toolView = document.getElementById('toolView');
  const toolContent = document.getElementById('toolContent');
  const backBtn = document.getElementById('backBtn');
  const logoutBtn = document.getElementById('logoutBtn');
  const searchInput = document.getElementById('searchInput');
  const categoryBtns = document.querySelectorAll('.category-btn');
  const toolItems = document.querySelectorAll('.tool-item');

  // Login elements
  const loginEmail = document.getElementById('loginEmail');
  const loginPassword = document.getElementById('loginPassword');
  const loginBtn = document.getElementById('loginBtn');
  const loginError = document.getElementById('loginError');

  // Update banner elements
  const updateBanner = document.getElementById('updateBanner');
  const dismissUpdate = document.getElementById('dismissUpdate');

  // Check for updates
  checkForUpdates();

  // Check if user is logged in
  checkAuth();

  // Update check function
  async function checkForUpdates() {
    try {
      // Check if user dismissed update for this version
      const dismissed = await new Promise(r => chrome.storage.local.get(['mnt_dismissed_version'], res => r(res.mnt_dismissed_version)));

      const res = await fetch(`${API_BASE}/extension-version.php?_=${Date.now()}`);
      const data = await res.json();

      if (data.version && data.version !== CURRENT_VERSION && data.version !== dismissed) {
        updateBanner.classList.remove('hidden');
      }
    } catch (e) {
      // Silent fail - don't show banner on error
    }
  }

  // Dismiss update banner
  if (dismissUpdate) {
    dismissUpdate.onclick = async () => {
      updateBanner.classList.add('hidden');
      // Remember dismissed version
      const res = await fetch(`${API_BASE}/extension-version.php`);
      const data = await res.json();
      if (data.version) {
        chrome.storage.local.set({ mnt_dismissed_version: data.version });
      }
    };
  }

  async function checkAuth() {
    const session = await getSession();
    if (session && session.token) {
      // Verify token is still valid
      try {
        const res = await fetch(`${API_BASE}/verify-session.php`, {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ token: session.token })
        });
        const data = await res.json();
        if (data.valid) {
          showToolsScreen();
          return;
        }
      } catch (e) {
        console.log('Session check failed, showing login');
      }
      // Token invalid, clear it
      await clearSession();
    }
    showLoginScreen();
  }

  function showLoginScreen() {
    loginScreen.classList.remove('hidden');
    toolsScreen.classList.add('hidden');
    toolView.classList.add('hidden');
    logoutBtn.classList.add('hidden');
    backBtn.classList.add('hidden');
    loginEmail.focus();
  }

  function showToolsScreen() {
    loginScreen.classList.add('hidden');
    toolsScreen.classList.remove('hidden');
    toolView.classList.add('hidden');
    logoutBtn.classList.remove('hidden');
    backBtn.classList.add('hidden');
    searchInput.focus();
  }

  // Login handler
  loginBtn.onclick = async () => {
    const email = loginEmail.value.trim();
    const password = loginPassword.value;

    if (!email || !password) {
      showError('Please enter email and password');
      return;
    }

    loginBtn.disabled = true;
    loginBtn.textContent = 'Signing in...';
    loginError.classList.add('hidden');

    try {
      const res = await fetch(`${API_BASE}/login.php`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ email, password })
      });
      const data = await res.json();

      if (data.success && data.token) {
        await saveSession({ token: data.token, email: email });
        showToolsScreen();
      } else {
        showError(data.error || 'Invalid email or password');
      }
    } catch (e) {
      showError('Connection failed. Please try again.');
    }

    loginBtn.disabled = false;
    loginBtn.textContent = 'Sign In';
  };

  // Allow Enter key to submit
  loginPassword.onkeydown = (e) => {
    if (e.key === 'Enter') loginBtn.click();
  };

  function showError(msg) {
    loginError.textContent = msg;
    loginError.classList.remove('hidden');
  }

  // Logout handler
  logoutBtn.onclick = async () => {
    await clearSession();
    showLoginScreen();
  };

  // Session storage helpers
  function getSession() {
    return new Promise((resolve) => {
      chrome.storage.local.get(['mnt_session'], (result) => {
        resolve(result.mnt_session || null);
      });
    });
  }

  function saveSession(session) {
    return new Promise((resolve) => {
      chrome.storage.local.set({ mnt_session: session }, resolve);
    });
  }

  function clearSession() {
    return new Promise((resolve) => {
      chrome.storage.local.remove(['mnt_session'], resolve);
    });
  }

  // Show tools list
  function showToolsList() {
    toolsScreen.classList.remove('hidden');
    toolView.classList.add('hidden');
    backBtn.classList.add('hidden');
    logoutBtn.classList.remove('hidden');
    searchInput.focus();
  }

  // Show specific tool
  function showTool(toolId) {
    const tool = Tools[toolId];
    if (!tool) return;

    toolsScreen.classList.add('hidden');
    toolView.classList.remove('hidden');
    backBtn.classList.remove('hidden');
    logoutBtn.classList.add('hidden');

    // Render tool content
    toolContent.innerHTML = `
      <div class="tool-header">
        <h2>${tool.title}</h2>
      </div>
      ${tool.render()}
    `;

    // Initialize tool
    if (tool.init) tool.init();

    // Setup copy buttons
    setupCopyButtons();
  }

  // Back button
  backBtn.onclick = showToolsList;

  // Tool item clicks
  toolItems.forEach(item => {
    item.onclick = () => {
      const toolId = item.dataset.tool;
      if (toolId) showTool(toolId);
    };
  });

  // Category filter
  categoryBtns.forEach(btn => {
    btn.onclick = function() {
      categoryBtns.forEach(b => b.classList.remove('active'));
      this.classList.add('active');
      filterTools();
    };
  });

  // Search filter
  searchInput.oninput = filterTools;

  function filterTools() {
    const searchTerm = searchInput.value.toLowerCase().trim();
    const activeBtn = document.querySelector('.category-btn.active');
    const category = activeBtn ? activeBtn.dataset.category : 'all';

    toolItems.forEach(item => {
      const name = item.querySelector('.tool-name').textContent.toLowerCase();
      const itemCategory = item.dataset.category;

      const matchesSearch = name.includes(searchTerm);
      const matchesCategory = category === 'all' || itemCategory === category;

      item.classList.toggle('hidden', !(matchesSearch && matchesCategory));
    });
  }
});
